import sqlite3
from contextlib import closing
from datetime import date, timedelta
import time
from flask import Flask, request, session, g, redirect, url_for, abort, render_template, flash

app = Flask(__name__)
app.config.from_object('todoo_config')

@app.route('/registeruser', methods=['POST'])
def registeruser():
    username = request.form['username']
    pwd = request.form['password']
    if not username:
        return render_template('signup.html', error="You must provide a username!", signup=True)
    elif not pwd:
        return render_template('signup.html', error="You must provide a password!", signup=True)
    elif query_db("select * from Users where username = ?", [username], one=True):
        return render_template('signup.html', error="Username already exists!", signup=True)
    else:
        d = get_connection()
        d.execute("insert into Users (username, pwd) values (?, ?)", [username, pwd])
        d.commit()
        flash("User '" + username + "' created")
        return redirect(url_for('login'))

@app.route('/signup')
def signup():
    return render_template('signup.html', signup=True)

@app.route('/login', methods=['GET', 'POST'])
def login():
    error = None
    if request.method == 'POST':
        username = request.form['username']
        user = query_db('select * from Users where username = ?', [username], one=True)
        if user is None:
            error = 'Invalid username!'
        elif user['pwd'] != request.form['password']:
            error = 'Invalid password!'
        else:
            session['user'] = user
            session['logged_in'] = True
            flash('Logged in')
            return redirect(url_for('view_week', year=current_year(), week=current_week()))
    return render_template('login.html', error=error)

@app.route('/logout')
def logout():
    session.pop('logged_in', None)
    flash('Logged out')
    return redirect(url_for('login'))

@app.route('/help')
def help():
    return render_template('help.html', help=True)

@app.route('/view')
def view():
    if not is_logged_in():
        return redirect(url_for('login'))
    now = date.today().isocalendar()
    return redirect('/view/' + str(now[0]) + '/' + str(now[1]))

@app.route('/view/<year>/<week>')
def view_week(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
        
    _year, week = refine_year_week(year, week)
    year = int(year)
    if _year != year: return redirect('/view/' + str(_year) + '/' + str(week))
    
    tasks = query_db("""select * from Tasks where user_id = ? and 
        (start_year < ? or (start_year = ? and start_week <= ?)) and 
        (end_year is null or (end_year > ? or (end_year = ? and end_week >= ?)))""",
    (session['user']['id'], year, year, week, year, year, week))
    
    view = (
    {'value': year, 'name': 'year', 'prevpath': '../' + str(year - 1) + '/' + str(week), 'nextpath': '../' + str(year + 1) + '/' + str(week)}, 
    {'value': week, 'name': 'week', 'prevpath': str(week - 1), 'nextpath': str(week + 1)}
    )
    return render_template('view.html', view=view, tasks=tasks)

@app.route('/view/<year>/<week>/add/', methods=['GET', 'POST'])
def add_task(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    d = get_connection()
    d.execute("insert into Tasks (user_id, description, start_year, start_week) values (?, ?, ?, ?)", 
    [session['user']['id'], str(request.args.get("desc")), year, week])
    d.commit()
    
    flash('Task added')
    return redirect('/view/' + str(year) + '/' + str(week))

@app.route('/view/<year>/<week>/remove/', methods=['GET', 'POST'])
def remove_task(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    d = get_connection()
    d.execute("delete from Tasks where user_id = ? and id = ?",
    [session['user']['id'], str(request.args.get("id"))])
    d.commit()
    
    flash('Task removed')
    return redirect('/view/' + str(year) + '/' + str(week))

@app.route('/view/<year>/<week>/check/')
def check_task(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    update_done(str(request.args.get("id")), 1, year, week)
    flash('Task checked')
    return redirect('/view/' + str(year) + '/' + str(week))

@app.route('/view/<year>/<week>/uncheck/')
def uncheck_done(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    update_done(str(request.args.get("id")), 0, None, None)
    flash('Task unchecked')
    return redirect('/view/' + str(year) + '/' + str(week))

def update_done(i, done, year, week):
    d = get_connection()
    d.execute("update Tasks set done = ?, end_year = ?, end_week = ? where user_id = ? and id = ?",
    [done, year, week, session['user']['id'], i])
    d.commit()

@app.route('/view/<year>/<week>/panic/')
def set_panic(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    update_panic(str(request.args.get("id")), 1)
    flash('Task set as panic')
    return redirect('/view/' + str(year) + '/' + str(week))

@app.route('/view/<year>/<week>/unpanic/')
def unset_panic(year, week):
    if not is_logged_in():
        return redirect(url_for('login'))
    update_panic(str(request.args.get("id")), 0)
    flash('Task no longer panic')
    return redirect('/view/' + str(year) + '/' + str(week))

def update_panic(i, panic):
    d = get_connection()
    d.execute("update Tasks set panic = ? where user_id = ? and id = ?",
    [panic, session['user']['id'], i])
    d.commit()

def refine_year_week(year, week):
    new_date = (date(int(year), 1, 1) + timedelta(weeks=int(week))).isocalendar()
    return (new_date[0], new_date[1])

def current_year():
    return date.today().year

def current_week():
    return int(date.today().strftime("%W"))

def is_logged_in():
    return session.has_key('logged_in') and session['logged_in']

@app.before_request
def before_request():
    g.db = connect_db()

@app.after_request
def after_request(response):
    g.db.close()
    return response

def get_connection():
    db = getattr(g, 'db', None)
    if db is None:
        db = g.db = connect_db()
    return db

def connect_db():
    return sqlite3.connect(app.config['DATABASE'])

def init_db():
    with closing(connect_db()) as db:
        with app.open_resource('schema.sql') as f:
            db.cursor().executescript(f.read())
        db.commit()

def query_db(query, args=(), one=False):
    cur = g.db.execute(query, args)
    rv = [dict((cur.description[idx][0], value)
               for idx, value in enumerate(row)) for row in cur.fetchall()]
    return (rv[0] if rv else None) if one else rv


if __name__ == '__main__':
    app.run()

