package DerpEngine;
import java.awt.*;
import java.awt.image.BufferStrategy;
import javax.swing.JFrame;

public class ScreenManager
{
	private GraphicsDevice device;
	private JFrame frame;
	private boolean isFullScreen;
	
	public ScreenManager()
	{
		GraphicsEnvironment environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
		device = environment.getDefaultScreenDevice();
	}
	
	public DisplayMode[] getCompatibleDisplayModes()
	{
		return device.getDisplayModes();
	}
	
	public DisplayMode findFirstCompatibleMode(DisplayMode[] modes)
	{
		DisplayMode goodModes[] = device.getDisplayModes();
		for(int i = 0; i < modes.length; i++)
		{
			for(int j = 0; j < goodModes.length; j++)
			{
				if(displayModesMatch(modes[i], goodModes[j]))
				{
					return modes[i];
				}
			}
		}
		return null;
	}
	
	public DisplayMode getCurrentDisplayMode()
	{
		return device.getDisplayMode();
	}
	
	public boolean displayModesMatch(DisplayMode mode1, DisplayMode mode2)
	{
		if(mode1.getWidth() != mode2.getWidth() || mode1.getHeight() != mode2.getHeight())
		{
			return false;
		}
		if(mode1.getBitDepth() != DisplayMode.BIT_DEPTH_MULTI && 
			mode2.getBitDepth() != DisplayMode.BIT_DEPTH_MULTI && 
			mode1.getBitDepth() != mode2.getBitDepth())
		{
			return false;
		}
		if(mode1.getRefreshRate() != DisplayMode.REFRESH_RATE_UNKNOWN &&
			mode2.getRefreshRate() != DisplayMode.REFRESH_RATE_UNKNOWN &&
			mode1.getRefreshRate() != mode2.getRefreshRate())
		{
			return false;
		}
		
		return true;
	}
	
	/*
	Enters full screen mode and changes the display mode.
	*/
	public void setDisplayMode(DisplayMode displayMode, boolean fullScreen)
	{
		
		frame = new JFrame();
		frame.setIgnoreRepaint(true);
		frame.setResizable(false);
		
		if(fullScreen)
		{
			frame.setUndecorated(true);
			device.setFullScreenWindow(frame);
			isFullScreen = true;
			if(displayMode != null && device.isDisplayChangeSupported())
			{
				try
				{
					device.setDisplayMode(displayMode);
				}
				catch(IllegalArgumentException ex)
				{
					isFullScreen = false;
				}
			}
		}
		else
		{
			isFullScreen = false;
			frame.setSize(displayMode.getWidth(), displayMode.getHeight());
			frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
			frame.setVisible(true);
		}
		
		frame.createBufferStrategy(2);
	}
	
	public Graphics2D getGraphics()
	{
		if(isFullScreen)
		{
			Window window = device.getFullScreenWindow();
			if(window != null)
			{
				BufferStrategy strategy = window.getBufferStrategy();
				return (Graphics2D)strategy.getDrawGraphics();
			}
			else
			{
				return null;
			}
		}
		else
		{
			return (Graphics2D)frame.getBufferStrategy().getDrawGraphics();
		}
		
	}
	
	public void update()
	{
		if(isFullScreen)
		{
			Window window = device.getFullScreenWindow();
			if(window != null)
			{
				BufferStrategy strategy = window.getBufferStrategy();
				if(!strategy.contentsLost())
				{
					getGraphics().dispose();
					strategy.show();
				}
			}
	        Toolkit.getDefaultToolkit().sync();
		}
		else
		{
			BufferStrategy strategy = frame.getBufferStrategy();
			if(!strategy.contentsLost())
			{
				getGraphics().dispose();
				strategy.show();
			}
		}
	}
	
	/*
	Returns the window currently used
	*/
	public Window getWindow()
	{
		if(isFullScreen)
		{
			return device.getFullScreenWindow();
		}
		else
		{
			return (Window)frame;
		}
	}
	
	public void closeWindow()
	{
		frame = null;
	}
	
	public void changeRes(int x, int y)
	{
		if(isFullScreen)
		{
			device.getFullScreenWindow().setBounds(frame.getX(), frame.getY(), x, y);
		}
		else
		{
			frame.setBounds(frame.getX(), frame.getY(), x, y);
		}
	}
	
	/*
	 * Returns the width of the window currently used
	 */
	public int getWidth()
	{
		if(isFullScreen)
		{
			Window window = device.getFullScreenWindow();
			if(window != null)
			{
				return window.getWidth();
			}
			else
			{
				return 0;
			}
		}
		else
		{
			return frame.getWidth();
		}
	}
	
	public int getHeight()
	{
		if(isFullScreen)
		{
			Window window = device.getFullScreenWindow();
			if(window != null)
			{
				return window.getHeight();
			}
			else
			{
				return 0;
			}
		}
		else
		{
			return frame.getHeight();
		}
	}
	/*
	Restores the screen's display mode. 
	*/
	public void restoreScreen()
	{
		if(isFullScreen)
		{
			Window window = device.getFullScreenWindow();
			if(window != null)
			{
				window.dispose();
			}
			device.setFullScreenWindow(null);
		}	
	}
	
}