package DerpEngine;
import java.awt.Color;
import java.awt.DisplayMode;
import java.awt.Graphics2D;
import java.awt.event.KeyListener;
import java.util.*;

public abstract class GameCore
{
	private static final DisplayMode POSSIBLE_DISPLAYMODES[] = {
		new DisplayMode(800, 600, 32, 0),
		new DisplayMode(800, 600, 24, 0),
		new DisplayMode(800, 600, 16, 0)
	};
	
	protected ScreenManager screen;
	protected GraphicsEngine gEngine;	
	private GameState currState;
	private boolean running = true;
	private boolean gameStateChanged = false;
	private boolean gamePaused = false;
	private double startTime;
	private double currTime;
	private double maxFps;
	private GameState changeToState;
	private boolean clearOldState;	
	private InputManager inputManager;
	private EventHandler eh;
	private Physics physics;
	private Loader loader;
	
	public void run()
	{
		try{
			initCore();
			currState = getGameStates().get(0);			
			currState.setGameCore(this);
			
			currState.setInputManager(inputManager);
			HashMap tempMap = new HashMap();
			Set<Map.Entry<Integer, GameAction>> set = getGameActions().entrySet();
			for(Map.Entry<Integer, GameAction> me : set)
			{
				inputManager.mapToKey(me.getValue(), me.getKey());
				tempMap.put(me.getValue().getName(), me.getValue());
			}
			currState.setGameActions(tempMap);			
			currState.initState(loader);			
			currState.setWorld(physics);

			gameLoop();
		}
		finally{
			System.out.println("RESTORING SCREEN");
			screen.restoreScreen();
			System.exit(0);
		}
	}
	//Initiate the subsystems.
	public void initCore()
	{
		screen = new ScreenManager();
		screen.setDisplayMode(screen.findFirstCompatibleMode(POSSIBLE_DISPLAYMODES), false);
		screen.getWindow().setFocusTraversalKeysEnabled(false);
		gEngine = new GraphicsEngine();
		
		inputManager = new InputManager(screen.getWindow());
		
		eh = getEventHandler();
		physics = new Physics();
		loader = new Loader();
		System.out.println("CORE INITIATED");
		init();
		System.out.println("GAME INITIATED");
		
		if(maxFps==0){
			maxFps = 1000/60;
		}
	}
	
	public void shutDown()
	{
		running =! running;
	}
	
	public void gameLoop()
	{
		startTime = (double)System.currentTimeMillis();
		currTime = startTime;
		System.out.println("GameCore gameLoop started "+startTime);
		while(running)
		{
			double elapsedTime = (double)System.currentTimeMillis() - currTime;
	
			if(gameStateChanged)
			{
				if(clearOldState)
				{
					currState.clearData();
				}
				
				removeListeners((KeyListener)currState);
				addListeners((KeyListener)changeToState);
				
				currState = changeToState;
				currState.initState(loader);
				
				
				gameStateChanged = false;
				
				elapsedTime = System.currentTimeMillis() - currTime;
				currTime += elapsedTime;
				elapsedTime = 0;
			}
			currTime += elapsedTime;
			

			//DRAWING BACKGROUND!
			Graphics2D g = screen.getGraphics();
			g.setColor(Color.black);
			g.fillRect(0, 0, screen.getWidth(), screen.getHeight());
			
			
			
			
			currState.checkGameInput();

			for(Sprite spr : currState.getGameObjects())
			{
				spr.Update(elapsedTime);
			}
								
			gEngine.drawSprites(currState.getGameObjects(),screen,currState.getFocusObject(),currState.getFocusOffset());//EDITED to send the screen since a reference to screen isnt set in the GraphicsEngine
			
			
			g.setColor(Color.white);
			int posY = 40;
			for(Sprite sprite : currState.getGameObjects()){
				if(sprite.debug()){
					g.drawString(""+sprite.getId()+" : "+sprite.getPosX()+" , "+sprite.getPosY()+" body : "+sprite.getBody().getBounds().getX()+" , "+sprite.getBody().getBounds().getY()+ " , force "+sprite.getForce().getX()+" , "+sprite.getForce().getY(), 20, posY);
					posY+=50;
				}
			}
			currState.runLoop(elapsedTime);
			
			
			physics.update(elapsedTime, currState.getGameObjects(), eh, maxFps);			
			screen.update();
			long sleepTime = (long)(maxFps-(System.currentTimeMillis() - currTime));
			if(sleepTime < 0){
				sleepTime = 0;
			}
			try{
				Thread.sleep(sleepTime);
			}catch(InterruptedException e){
			}
		}
	}
	public void setMaxFps(double targetFPS){
		maxFps = 1000/targetFPS;
	}
	public void removeListeners(KeyListener listener)
	{
		screen.getWindow().removeKeyListener(listener);
	}
	
	public void addListeners(KeyListener listener)
	{
		screen.getWindow().addKeyListener(listener);
	}
	
	public synchronized void changeGameState(String strState, boolean clearOldState)
	{
		System.out.println("changeGameState (GameCore)");
		for(GameState state : getGameStates())
		{
			if(strState.equals(state.getName()))
			{
				changeToState = state;
				this.clearOldState = clearOldState;
				gameStateChanged = true;
			}
		}
	}
	
	public abstract ArrayList<GameState> getGameStates();
	public abstract EventHandler getEventHandler();

	public abstract Map getGameActions();
	public abstract void init();

}